  
%macro fitdistrib(data,stratum,id,repeat,baseweight,weightvarroot,episodic,daily,covlisti,
                  covlistc,sequence,weekend,outstd,tranout,replicfirst,repliclast,
                  n_mcmc_iterations,n_burn_iterations,thin,folder,outpath,gen_inverse,
                  print,ntitles,store_sd, notes_print_mcmc,npseudopeople,seed);
                 
  
  /**************************************************************************/
  /*** the fitdistrib macro is used in conjunction with the HEI Wrapper   ***/
  /*** example of how to fit an MCMC model using NHANES data and BRR      ***/
  /*** weights to obtain distributions. Steps 4-8 of the process are      ***/
  /*** executed in this macro.                                            ***/
  /***                                                                    ***/
  /*** The steps are:                                                     ***/
  /***                                                                    ***/
  /*** STEP 4: FIND MINIMUM VALUES                                        ***/
  /*** Find the minimum value for all components identified as daily by   ***/
  /*** stratum                                                            ***/
  /***                                                                    ***/
  /*** STEP 5: FIND LAMBDAS                                               ***/
  /*** This step identifies the best Box Cox transformation               ***/
  /*** for each dietary component by stratum                              ***/
  /***                                                                    ***/
  /*** STEP 6: TRANSFORM DATA TO STANDARD NORMAL                          ***/
  /*** This step runs the std_cov_boxcox24hr_conday_minamt_macro_v1.0.sas ***/
  /*** macro to create the input data set.                                ***/
  /***                                                                    ***/
  /*** STEP 7: RUN MCMC MODEL FITTING                                     ***/
  /*** This is the most time consuming step. For NHANES 11/12 it takes    ***/
  /*** about 1 hour to run 1 replicate for 1 stratum using 5000 mcmc      ***/
  /*** iterations and 3000 burnins                                        ***/
  /*** Note: Because this is so time consuming we recommend running the   ***/
  /*** base replicate (0) first, then the other replicates                ***/
  /***                                                                    ***/
  /*** STEP 8: RUN DISTRIBUTION MACRO                                     ***/
  /*** This macro will create the multivariate distributions of usual     ***/
  /*** intake for all of the dietary components                           ***/
  /**************************************************************************/
  
  /**************************************************************************/
  /*** The parameters in the macro fitdistrib are passed to parameters    ***/ 
  /*** in one or more of the macros that are called, and/or used in the   ***/
  /*** body of fitdistrib to prepare the data.                            ***/
  /**************************************************************************/
  
/*********************************************************************************************************/
/***           the parameters in the fitdistrib macro are used as follows.                             ***/
/***                                                                                                   ***/
/*** data             =   the name of the data set prepared in steps 1-3.                              ***/
/***                      Used in fitdistrib(getlambda,getminu),std_cov_boxcox24hr_conday_minamt       ***/
/***                                                                                                   ***/
/*** stratum          =   the variable that defines the strata level.                                  ***/
/***                      values shoud be integers. If no strata are selected the value must be 1.     ***/
/***                      Used in fitdistrib(getlambda,getminu), std_cov_boxcox24hr_conday_minamt,     ***/
/***                      multivar_distrib                                                             ***/
/***                                                                                                   ***/
/*** id               =   A variable that defines a unique subject identifier and is passed to the     ***/
/***                      multivar_mcmc parameter subject.                                             ***/
/***                      Used in  multivar_mcmc                                                       ***/
/***                                                                                                   ***/
/*** repeat           =   Specifies a variable that indexes repeated observations for each subject.    ***/
/***                      Used in multivar_mcmc                                                        ***/
/***                                                                                                   ***/
/*** baseweight       =   The name of the original sample weight variable ending in _0. (e.g. Repwt_0) ***/
/***                      Used in fitdistrib(getlambda)                                                ***/
/***                                                                                                   ***/
/*** weightvarroot    =   The root name for the base and other weights used for SE such as             ***/
/***                      BRR weights (e.g. Repwt_). This is passed to the multivar_mcmc               ***/
/***                      macro parameter weight_var and used in conjunction                           ***/
/***                      with the replicate number.                                                   ***/
/***                      Used in multivar_mcmc                                                        ***/
/***                                                                                                   ***/
/*** episodic         =   The names of the dietary components consumed episodically                    ***/
/***                      Used in fitdistrib, std_cov_boxcox24hr_conday_minamt                         ***/
/***                                                                                                   ***/
/*** daily            =   The names of the dietary components that are defined daily                   ***/
/***                      Used in fitdistrib, std_cov_boxcox24hr_conday_minamt                         ***/
/***                                                                                                   ***/
/*** covlisti         =   The names of the covariates that are indicator (dummy) variables             ***/
/***                      Used in fitdistrib and also passed to several                                ***/
/***                      paramters in the multivar_mcmc and multivar_distrib macros.                  ***/
/***                      Used in fitdistrib(getlamba,getminu), multivar_mcmc, mulitvar_distrib        ***/
/***                                                                                                   ***/
/*** covlistc         =   The names of the covariates that are continuous variables                    ***/
/***                      Used in fitdistrib and also passed to several                                ***/
/***                      paramters in the multivar_mcmc and multivar_distrib macros.                  ***/
/***                      Used in fitdistrib(getlamba,getminu), multivar_mcmc, mulitvar_distrib        ***/
/***                                                                                                   ***/
/*** sequence         =   An indicator variable that indicates whether the recall is from              ***/
/***                      day 1 or not. Specifically, it is coded 1 if day=2 and coded 0               ***/
/***                      if day=1.                                                                    ***/
/***                      Used in multivar_mcmc                                                        ***/
/***                                                                                                   ***/
/*** weekend          =   An indicator variable that is set to 1 for weekends and 0 for                ***/
/***                      weekdays.                                                                    ***/
/***                      Used in multivar_mcmc                                                        ***/
/***                                                                                                   ***/
/*** outstd           =   A user supplied name for a data set output from the                          ***/
/***                      std_cov_boxcox24hr_conday_minamt macro                                       ***/
/***                      and passed to the data parameter of the multivar_mcmc macro                  ***/
/***                      that contains the standardized [N(0,1)] varaibles.                           ***/
/***                      Saved in std_cov_boxcox24hr_conday_minamt; Used in multivar_mcmc             ***/
/***                                                                                                   ***/
/*** tranout          =   A user supplied name for a dataset output from the                           ***/ 
/***                      std_cov_boxcox24hr_conday_minamt macro                                       ***/
/***                      and passed to the optional_iml_store_data parameter of the                   ***/
/***                      multivar_mcmc macro that contains the lamdas for the Box-Cox                 ***/
/***                      transformation (and backtransformation).                                     ***/
/***                      Saved in std_cov_boxcox24hr_conday_minamt; Used in multivar_mcmc             ***/
/***                                                                                                   ***/
/*** replicfirst      =   A number for the first replicate fit in the multivar_mcmc and                ***/
/***                      multivar_distrib macros using weights (in a loop from                        ***/
/***                      replicfirst to repliclast)                                                   ***/
/***                      Used in fitdistrib                                                           ***/
/***                                                                                                   ***/
/*** repliclast       =   A number for the last replicate fit in the multivar_mcmc and                 ***/
/***                      multivar_distrib macros using weights (in a loop from                        ***/
/***                      replicfirst to repliclast)                                                   ***/ 
/***                      Used in  fitdistrib                                                          ***/
/***                                                                                                   ***/
/*** n_mcmc_iterations=   The total number of iterations fro the MCMC algorithm                        ***/
/***                      including the burn-in                                                        ***/
/***                      Used in multivar_mcmc                                                        ***/
/***                                                                                                   ***/
/*** n_burn_iterations=   Specifies the burn-in value for the MCMC algorithm                           ***/
/***                      Used in multivar_mcmc                                                        ***/
/***                                                                                                   ***/
/*** thin             =   Specifies the thinning value for the MCMC algorithm                          ***/
/***                      Used in multivar_mcmc                                                        ***/
/***                                                                                                   ***/
/*** folder           =   File reference to define a folder for the multivar_mcmc output.              ***/
/***                      Used in fitdistrib                                                           ***/
/***                                                                                                   ***/
/*** outpath          =   Defines the libname for output                                               ***/
/***                      used in fitdistrib                                                           ***/
/***                                                                                                   ***/
/*** gen_inverse      =   If this is defined as "n" or "N" then the SAS IML function "inv" is used     ***/
/***                      to compute the inverse of matrices in the MCMC algorithm. It may save        ***/
/***                      time but the user may encounter an error that the matrix should be           ***/
/***                      non-singular. By default the "ginv" is used to compute the Moore-Penrose     ***/
/***                      generalized inverse of matrices needed in the MCMC algorithm.                ***/
/***                      Used in multivar_mcmc                                                        ***/
/***                                                                                                   ***/
/*** print            =   Determines whether macro results are printed                                 ***/
/***                      Used in std_cov_boxcox24hr_conday_minamt, multivar_mcmc, multivar_distrib    ***/
/***                                                                                                   ***/
/*** ntitles          =   Specifies the number of title lines to be reserved for the user's titles     ***/
/***                      Used in multivar_mcmc                                                        ***/
/***                                                                                                   ***/
/*** store_sd         =   Determines whether sample standard deviations are printed and stored in      ***/
/***                      the IML storage catalog.                                                     ***/
/***                      Used in multivar_mcmc                                                        ***/
/***                                                                                                   ***/
/*** notes_print_mcmc =   Determines whether the notes are printed to the SAS log                      ***/
/***                      Used in multivar_mcmc                                                        ***/
/***                                                                                                   ***/
/*** npseudopeople    =   The number of pseudo-people used to estimate the distribution of intake      ***/
/***                      Used in multivar_distrib                                                     ***/
/***                                                                                                   ***/
/*** seed             =   A starting seed used by the fitdistrib macro before calling the              ***/
/***                      multivar_mcmc and multivar_distrib macros.                                   ***/
/***                      Used in fitdistrib                                                           ***/
/*********************************************************************************************************/
 
  libname liblabel "&liblabel"; 
  
  /* create a macro variable listing all the dietary components */
  
  %let list=&episodic &daily; 
  
  
  *************************
  *** STEP 4 and STEP 5  **
  *************************;
  
  %put Starting STEP 4: FIND MINIMUM VALUES    ;
  ********************************************************************************
  ** define macros getminu and getlambda to find minimum positive value of      **
  ** variables consumed daily and the best lambda value for each dietary        **
  ** component.                                                                 **
  ********************************************************************************;
  
  *********************************************************
  ** define getminu macro to find the min non-zero value **
  ** (This will be used in step 4.                       **
  *********************************************************;
  %macro getminu (data,dvar,num);
  
    /***           ---- Paramater Description ----                 ***/             
    /*  data	  The name of the input data set.                      */
    /*          This parameter is defined in the call to fitdistrib. */
    /*  dvar    Variable name of dietary components consumed daily.  */
    /*          The value is created in the call to this macro.      */
    /*  num  =  The current position of variable names in "list".    */
    /*          The value is created in the call to this macro.      */
    /***                                                           ***/
    ;
    
    proc means data=&data noprint ;
    by stratum;
    where &dvar > 0;
    var &dvar;
    output out=minu1 min=minamount;
    run;
  
    data minu; format component $30.; set minu1; component="&dvar"; tran_paramindex=&num; minamount=minamount*0.5; run;
  
    proc append base=minvalu data=minu; run;
  
    proc datasets nolist; delete minu; run;
  
  %mend getminu;
  
  %put Starting STEP 5: FIND LAMBDAS ;
  ************************************************
  ** define getlambda macro to find the lambdas **
  ** This will be used in step 5                **
  ************************************************;
  %macro getlambda (datam,invar,covlistc,covlisti,baseweight,vnum);
  
    /***             ---- Paramater Description ----                ***/
    /*  data	      Defined in the macro fitibit.                     */
    /*  invar       The variable name of each dietary component.      */
    /*              The value is extracted in the call to this macro  */   
    /*  covlistc    Defined in the macro fitdistrib.                  */
    /*  covlisti    Defined in the macro fitdistrib.                  */
    /*  baseweight  Defined in the macro fitdistrib.                  */ 
    /*  vnum        The value is created in the call to the macro.    */         
    /***                                                            ***/
    ;
    
    ods graphics off;
    ** turn off printing ** ;
      ods exclude all;  
      
    ** define the identity variables **;   
      %let ident=%str(&covlistc &covlisti) ;
      
      proc transreg data=&datam pboxcoxtable; by stratum;
            where &invar>0;
            model BoxCox(&invar / lambda=0.01 to 1 by 0.01) = identity(&ident); 
  		     freq &baseweight;
  		     ods output boxcox=boxcox;
      run;
  
      data select;
            set boxcox(rename=(lambda=tran_lambda));
            if ci='<';
  		  component=substr(dependent,8,(index(dependent,')')-8));
     	      tran_paramindex=&vnum;
  		  keep component stratum tran_paramindex tran_lambda;
      run;
  
      proc append base=lambdas data=select; run;
  
  ods graphics on;
  ** turn on printing ** ;
  ods select all;
  
  proc datasets nolist; delete select; run;
  
  %mend getlambda;
  
  *******************************************************************;
  
     
  title%eval(&titles+2) "STEPS 4 & 5: FIND MINIMUM VALUES & LAMBDAS " ; run;
  
  run;
  
  %* Count the number of words in &LIST;
  %local count;
  %let count=0; 
  %do %while(%qscan(&list,&count+1,%str( )) ne %str());
  %let count = %eval(&count+1); 
  %end;
  
  *******************************************
  ** call the minimum amount macro         **
  *******************************************;
  %put Finding minimum and lambdas for stratum &stratum;
  
  %do i = 1 %to &count;
  %getminu(&data,%scan(&list,&i),&i)
  %end;
  
  title%eval(&titles+3) 'Minimum Values';
  proc print data=minvalu;
  run;


  *******************************************
  ** create dataset with min values        **
  *******************************************;

    * Count the number of words in &daily;
  %local countd;
  %let countd=0; 
  %do %while(%qscan(&daily,&countd+1,%str( )) ne %str());
  %let countd = %eval(&countd+1); 
  %end;

  proc transpose data=minvalu out=minvalut prefix=min; var minamount; id component; run;

  data datamin;
  if _n_=1 then set minvalut(drop=_NAME_ _LABEL_);
	set &data;

  %macro min(dvar);
  if &dvar = 0 then do; &dvar=0.5*min&dvar; end;
  %mend min;

  %do i = 1 %to &countd;
  %min(%scan(&daily,&i))
  %end;

  run;

  *******************************************
  ** call the lambda macro                 **
  *******************************************;

  %do i = 1 %to &count;
  %getlambda(datamin,%scan(&list,&i),&covlistc,&covlisti,&baseweight,&i)
  %end;

  title%eval(&titles+3) 'Lambdas';
  proc print data=lambdas; 
  run;
 
  
  ****************************************************;
  *** STEP 6: TRANSFORM DATA TO STANDARD NORMAL    ***;
  ****************************************************;
  
  title%eval(&titles+2) "STEP 6: TRANSFORM DATA TO STANDARD NORMAL" ; 
  run;
  
  %put Starting STEP 6: TRANSFORM DATA TO STANDARD NORMAL ;
  
  ***********************************************************
  *** Call the SAS macro std_cov_boxcox24hr_conday_minamt ***  
  *** for the first iteration only.                       ***
  ***********************************************************;
  
  %if &replicfirst = 0 %then %do ;
  
    %put Standardizing covariates for stratum &stratum;
    
    %std_cov_boxcox24hr_conday_minamt(data                            = &data, 
                                      prestand_continuous_covars      = &covlistc,
                                      rec24hr_epis_vars               = &episodic,
                                      rec24hr_daily_vars              = &daily,
                                      boxcox_tran_lambda_data         = lambdas,   
                                      minamount_data                  = minvalu,   
                                      print                           = y,
                                      titles                          = 3 ); 
    
    data  &outstd;
    set stdcov_stdbc24hr_conday_out;
    run;
    
    data &tranout;
    set backtran_out;
    run;
  
  %end;   /* of creating standaridized covariates  */ ;  
  
  proc datasets nolist; delete lambdas minvalu; run;
  
  /*****************************************************************************/
  /* Preparation for steps 7 & 8                                               */
  /*****************************************************************************/
  
  /*****************************************************************************/
  /* The macro variables are that are needed for the MULTIVAR_MCMC macro will  */
  /* use are printed:                                                          */
  /*     conday_var_list                                                       */
  /*     stdbc_epis_var_list                                                   */
  /*     stdbc_daily_var_list                                                  */
  /*     std_continuous_covar_list.                                            */
  /*****************************************************************************/
  
   %put Variables from the std_cov_boxcox24hr_conday_minamt (Stratum &stratum);
   %put consumption day variables: &conday_var_list;
   %put standardized episodic variables: &stdbc_epis_var_list;
   %put standardized daily variables: &stdbc_daily_var_list;
   %put standardized continuous variables: &std_continuous_covar_list;
  
  ****************************************************************
  *** call the SAS macro multivar_mcmc & distrib for each      ***
  *** replicate                                                *** 
  ****************************************************************;
  
  
  **************************************;
  *** STEP 7: RUN MCMC MODEL FITTING ***;
  **************************************;
  
  title%eval(&titles+2)  'STEP 7: RUN MCMC MODEL FITTING ';
  title%eval(&titles+3)  "MCMC Model fitting using the mulitvar_mcmc macro";
  run;
  
  %put Starting STEP 7: RUN MCMC MODEL FITTING ;
  
  
  %let example_stratum_label = &folder;   ** folder for output data ** ;
   
  %macro fit_models_replicate_loop(); 
  
  %do replicnum = &replicfirst %to &repliclast;
  
  %put MCMC Stratum: &stratum Replicate: &replicnum;
    
      %let loopseed = %eval(&seed + &replicnum * 1000); 
      %let outlabel = &example_stratum_label._replic&replicnum;
      %let graph_pdf_name = &liblabel.&example_stratum_label..trace.replic&replicnum..pdf;

  %put Output Plot Location &graph_pdf_name;
  
  /****************************************************/   
  /*** Fit the multivariate measurement error model ***/ 
  /****************************************************/
  
  %multivar_mcmc    (data                        = &outstd,
                     subject                     = &id,
                     weight_var                  = &weightvarroot.&replicnum,  
                     repeat                      = &repeat,
                     conday_epis_vars            = &conday_var_list,
                     gst_rec24hr_epis_vars       = &stdbc_epis_var_list,
                     gst_rec24hr_daily_vars      = &stdbc_daily_var_list,
                     covars_epis_prob            = constant1 &sequence &weekend &covlisti &std_continuous_covar_list,
                     covars_epis_amt             = constant1 &sequence &weekend &covlisti &std_continuous_covar_list,
                     covars_daily_amt            = constant1 &sequence &weekend &covlisti &std_continuous_covar_list,
                     set_seed_mcmc               = &loopseed,
                     set_number_mcmc_iterations  = &n_mcmc_iterations,
                     set_number_burn_iterations  = &n_burn_iterations,
                     set_thin                    = &thin,  
                     gen_inverse                 = &gen_inverse,
                     print                       = &print,
                     titles                      = &ntitles,
                     std_print_store             = &store_sd,
                     notes_print                 = &notes_print_mcmc  ,
                     out_lib                     = liblabel,
                     out_store_label             = &outlabel,
                     set_number_saved_out_data   = 0,
                     traceplots_method1_gpath    = ,
                     traceplots_method2_file_pdf = &graph_pdf_name,
                     optional_iml_store_data     = &tranout,
                     optional_iml_store_names    = constant1 &sequence &weekend &covlisti &std_continuous_covar_list
                                                   tran_paramindex tran_lambda tran_center tran_scale minamount        
                     );                 
  
  option nonotes ;
  **************************************;
  *** STEP 8: RUN DISTRIBUTION MACRO ***;
  **************************************;
  
  title%eval(&titles+2) 'STEP 8: RUN DISTRIBUTION MACRO'; 
  title%eval(&titles+3) 'Run distribution model using the macro multivar_distrib';
  title%eval(&titles+3) "Stratum &stratum  Replicate &replicnum";
  run;
  
  %put starting STEP 8: RUN DISTRIBUTION MACRO;
  
  *************************************************************************
  *** Call the MULTIVAR_DISTRIB macro where covariate list 1 and        ***
  *** covariate list 2 are specified for week days and weekends,        ***
  *** respectively.  Note that the multivariate measurement error model ***
  *** was fit using the MULTIVAR_MCMC macro with the following          ***
  *** covariates:                                                       ***
  ***   constant1, recallday2, weekend, and other covariates            ***
  *************************************************************************;
  
  %put Distrib Stratum &stratum Replicate &replicnum;
  
  %let loopseed_distrib = %eval(&seed + 10 + &replicnum * 1000); 
      
  %multivar_distrib(multivar_mcmc_out_lib           = liblabel,
                    multivar_mcmc_out_store_label   = &outlabel, 
                    t_weightavg_covariates_list1    = constant1 constant0 constant0 &covlisti &std_continuous_covar_list, 
                    t_weightavg_covariates_list2    = constant1 constant0 constant1 &covlisti &std_continuous_covar_list, 
                    set_value_for_weight_cov_list1  = 4,
                    set_value_for_weight_cov_list2  = 3,
                    set_seed_distrib                = &loopseed_distrib, 
                    set_number_monte_carlo_rand_obs = &npseudopeople, 
                    print                           =                             
                   );
    		          
  data liblabel.distmc&stratum.&replicnum;
  set mc_t_distrib_out;
  run;
      
  %end; /*** End loop for replicates ***/    
  
  %mend fit_models_replicate_loop;
  
  %fit_models_replicate_loop()


  %put Done!;
  
  %mend fitdistrib;
